import { showScreen } from "./main.js";
import { setDB, ScoreRecord, Scores } from "./db.js";

function execGame(limitSeconds = 3, difficulty = "easy"){
    const timeouts = [];
    const intervals = [];
    const rafs = [];
    const eventRemovers = [];
    let isPaused = false;

    function startGameLoop(limitSeconds) {
        const mainArea = document.getElementById("main-area");
        const counter = document.getElementById("counter");
        const dial = document.getElementById("meter-arc");
        const lifeView = document.getElementById("lives");
        const success = document.getElementById("success");
        const failed = document.getElementById("failed");
        const radius = 50;
        let lives = 3;
    
        counter.innerText = 1;
        let currentCount = parseInt(counter.innerText, 10);
    
        function polarToCartesian(cx, cy, r, angleDeg) {
            const rad = (angleDeg - 90) * Math.PI / 180;
            return {
                x: cx + r * Math.cos(rad),
                y: cy + r * Math.sin(rad)
            };
        }
    
        function describeArc(cx, cy, r, startAngle, endAngle) {
            const start = polarToCartesian(cx, cy, r, endAngle);
            const end = polarToCartesian(cx, cy, r, startAngle);
            const largeArcFlag = endAngle - startAngle <= 180 ? "0" : "1";
            return `M ${start.x} ${start.y} A ${r} ${r} 0 ${largeArcFlag} 0 ${end.x} ${end.y}`;
        }
    
        function drawLives(){
            lifeView.textContent = "";
            for(let i = 0; i < lives; i++){
                lifeView.textContent += "♥";
            }
        }
    
        function checkButtons() {
            const buttons = document.querySelectorAll('#buttons input[type="checkbox"]');
            for(let i = 0; i < buttons.length; i++){
                /**@type {HTMLInputElement} */
                const btn = buttons[i];
                const id = btn.id;
                if(id === "fizz"){
                    if(currentCount % 3 === 0){
                        if(!btn.checked) return false;
                    }
                    else{
                        if(btn.checked) return false;
                    }
                }
                if(id === "buzz"){
                    if(currentCount % 5 === 0){
                        if(!btn.checked) return false;
                    }
                    else{
                        if(btn.checked) return false;
                    }
                }
            }
            return true;
        }
    
        function clearCheck() {
            const buttons = document.querySelectorAll('#buttons input[type="checkbox"]');
            for(let i = 0; i < buttons.length; i++){
                /**@type {HTMLInputElement} */
                const btn = buttons[i];
                btn.checked = false;
            }
        }
    
        function checkResult(){
            mainArea.classList.remove("mode-count");
            mainArea.classList.add("mode-result");
            clearDisplay();
            if(checkButtons()){
                success.classList.add("display");
            }
            else{
                success.classList.remove("display");
                failed.classList.add("display");
                lives--;
            }
        }
    
        function resetLoop(){
            mainArea.classList.add("mode-count");
            clearDisplay();
            clearCheck();
            drawLives();
        }

        function clearDisplay(){
            failed.classList.remove("display");
            success.classList.remove("display");
        }
        
        function showGameOver(){
            clearDisplay();
            drawLives();
            mainArea.classList.remove("mode-result");
            mainArea.classList.add("mode-start");
            renderCountdownText(`GameOver!\nSCORE: ${currentCount}`);
        }

        function recordeScore(score){
            const record = new ScoreRecord(score);
            //スコア記録処理
            setDB(record, difficulty);
        }
    
        drawLives();
    
        function loop() {
            resetLoop();
            
            let gameElapsed = 0;       // ゲーム内で進行した時間（秒）
            let lastFrameTime = null;  // 前回フレーム時のtimestamp
            unPause = () => {
                lastFrameTime = performance.now();
            }

            function updateDial(now) {
                
                if(lastFrameTime === null) lastFrameTime = now;

                if(isPaused){ 
                    const fid = requestAnimationFrame(updateDial);
                    rafs.push(fid);
                    return;
                }
                else{
                    const delta = (now -lastFrameTime) /1000;
                    gameElapsed += delta;
                    lastFrameTime = now;
                    
                    const progress = Math.min(gameElapsed / limitSeconds, 1);
                    const endAngle = 360 * (1 - progress);
                    dial.setAttribute("d", describeArc(60, 60, radius, 0, endAngle));
        
                    if (progress < 1) {
                        const fid = requestAnimationFrame(updateDial);
                        rafs.push(fid);
                    } else {
                        // Step 2: Show result for 1 second
                        checkResult();
        
                        if(lives > 0){
                            const tid = setTimeout(() => {
                                // Step 3: Next count
                                currentCount++;
                                counter.innerText = currentCount;
                                mainArea.classList.remove("mode-result");
                                loop(); // Repeat loop
                            }, 1000);
                            timeouts.push(tid);
                        }
                        else{
                            showGameOver();
                            recordeScore(currentCount);
                        }
                    }
                }
            }
            const fid = requestAnimationFrame(updateDial);
            rafs.push(fid);
        }
        loop();
    }
    
    function renderCountdownText(count){
        document.getElementById("start-counter").textContent = count;
    }
    
    function showStartCountdown(onComplete) {
      let count = 3;
      const countdownInterval = setInterval(() => {
        if (count > 0) {
          renderCountdownText(count); // 3→2→1 を描画
          count--;
        } else {
          renderCountdownText("Start!");
          clearInterval(countdownInterval);
          const tid = setTimeout(() => {
            onComplete();
          }, 500); // “Start!”表示の余韻
          timeouts.push(tid);
        }
      }, 1000); // 1秒間隔
      intervals.push(countdownInterval);
    }
    
    function startGame() {
      showStartCountdown(() => {
        document.getElementById("main-area").classList.remove("mode-start");
        startGameLoop(limitSeconds);
      });
    }

    function dispose(){
        timeouts.forEach(tid => { if (tid) clearTimeout(tid); });
        intervals.forEach(iid => { if (iid) clearInterval(iid); });
        rafs.forEach(fid => { if (fid) cancelAnimationFrame(fid); });
        eventRemovers.forEach(disposer => disposer?.());
        document.getElementById("exit").onclick = () => {};
    }

    let unPause = () => {

    }
    
    document.getElementById("exit").onclick = () => {
        isPaused = true;
        if(confirm("ゲームを終了しますか？")){
            dispose();
            //タイトル画面に戻る処理
            showScreen("title");
        }else{
            isPaused = false;
            unPause();
        }
    }
    startGame();
}
export {execGame};
// //実際にはタイトルからの呼び出しで実行され、引数もタイトル側で設定した値が使われる。
// execGame(3);