// db.js
let db = null;

/** 得点記録
 * @class
 */
export class ScoreRecord {
  /** 得点。最終到達カウント値
   * @type {number}
   */
  score = 0;
  /** 得点記録時刻
   * @type {Date}
   */
  timestamp = Date.now();
  /** コンストラクタ
   * @param {number} s 得点
   */
  constructor(s){
    this.score = s;
  }
}

/** 難易度ごとの上位3位を記録するためのクラス
 * @class
 */
export class Scores {
  /** 難易度
   * @type {string}
   */
  difficulty = "";
  /** 上位3位の記録
   * @type {Array<ScoreRecord>}
   */
  top3 = [];
  constructor(d){
    this.difficulty = d;
  }
  /**
   * 
   * @param {Scores} json 
   */
  fromPlain(json){
    this.difficulty = json.difficulty;
    this.top3 = json.top3;
  }
  /** 新しい得点記録が上位3位内に食い込む場合にtop3を更新する処理
   * @param {ScoreRecord} record 
   */
  updateScore(record){
    // 現在のtop3に新記録を加える
    this.top3.push(record);
    // スコア順で降順にソート
    this.top3.sort((a, b) => b.score - a.score);
    // 上位3件のみ残す
    this.top3 = this.top3.slice(0, 3);
  }
}

export function initDB(callback) {
  const req = indexedDB.open("FizzBuzzDB", 2);
  req.onupgradeneeded = function(e) {
    const db = e.target.result;
    db.createObjectStore("scores", { keyPath: "difficulty", autoIncrement: false });
  };
  req.onsuccess = function(e) {
    db = e.target.result;
    callback(db);
  };
}

export function getDB() {
  return new Promise((resolve, reject) => {
    const req = indexedDB.open("FizzBuzzDB", 2);
    req.onupgradeneeded = function(e) {
      const db = e.target.result;
      db.createObjectStore("scores", { keyPath: "difficulty", autoIncrement: false });
    };
    req.onerror = () => reject(req.error);
    
    req.onsuccess = function(e) {
      const db = e.target.result;
      const tx = db.transaction("scores", "readwrite");
      const store = tx.objectStore("scores");

      const getReq = store.getAll();

      getReq.onsuccess = () => {
        resolve(getReq.result);
        if (getReq.result) {
          //getReq.result内容をgetDBの戻り値として返却したい
        } else {
          //取得内容がなかったということをgetDBの戻り値として返却したい
        }
      };
      getReq.onerror = () => reject(getReq.error);
    };
  });
}

export function setDB(score, difficulty) {
  const req = indexedDB.open("FizzBuzzDB", 2);
  req.onupgradeneeded = function(e) {
    const db = e.target.result;
    db.createObjectStore("scores", { keyPath: "difficulty", autoIncrement: false });
  };
  req.onsuccess = function(e) {
    const db = e.target.result;
    const tx = db.transaction("scores", "readwrite");
    const store = tx.objectStore("scores");

    const getReq = store.get(difficulty);

    getReq.onsuccess = () => {
      let entry = new Scores(difficulty);
      if (getReq.result) {
        // 既存データあり → Scoresクラスでtop3を更新
        entry.fromPlain(getReq.result);  // ← JSON → クラス変換（想定）
        entry.updateScore(score);              // ← クラスメソッドで更新
      } else {
        // 既存データなし → 新規でtop3に代入
        entry.top3.push(score);
      }

      store.put(entry); // 上書き or 新規挿入
    };

    const scoreEntry = {
      difficulty: difficulty, // ← 主キー
      top3: [
        score
      ]
    };

    // `difficulty` が存在すれば上書き、なければ新規挿入
    store.put(scoreEntry);

    tx.oncomplete = () => {
      console.log("保存完了！");
    };
  };
}
